﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Hims.Shared.UserModels;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{
    /// <inheritdoc />
    [Authorize]
    [Route("api/working-hour")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class WorkingHourController : BaseController
    {
        /// <summary>
        /// The working Hour  services.
        /// </summary>
        private readonly IWorkingHourService workingHourService;
        /// <summary>
        /// The AuditLog  services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;


        /// <inheritdoc />
        public WorkingHourController(IWorkingHourService workingHourService, IAuditLogService auditLogServices)
        {
            this.workingHourService = workingHourService;
            this.auditLogServices = auditLogServices;
           
        }

        /// <summary>
        /// Inserts the  asynchronous.
        /// </summary>
        /// <param name="workingHourModel">The model.</param>
        /// <param name="header">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> InsertAsync([FromBody] WorkingHourModel workingHourModel, [FromHeader] LocationHeader header)
        {
              
                workingHourModel = (WorkingHourModel)EmptyFilter.Handler(workingHourModel);
                var res = await this.workingHourService.InsertAsync(workingHourModel);
                var auditLogModel = new AuditLogModel
                {
                    AccountId = workingHourModel.CreatedBy,
                    LogTypeId = (int)LogTypes.WorkingHour,
                    LogFrom = (short)workingHourModel.LoginRoleId,
                    LogDate = DateTime.Now,
                    LogDescription = $"<b>{workingHourModel.CreatedByName}</b> has added <b>WorkingHour</b> of <strong>{workingHourModel.StartDate}</strong> successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };

                await this.auditLogServices.LogAsync(auditLogModel);
                return this.Success(res);
               
                       
                    }
        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-all")]
        public async Task<ActionResult> FetchAsync([FromBody] WorkingHourModel model)
        {
            model = (WorkingHourModel)EmptyFilter.Handler(model);
            var response = await this.workingHourService.FetchAllAsyn(model);
            return this.Success(response);

        }
        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] WorkingHourModel model, [FromHeader] LocationHeader header)
        {
            model = (WorkingHourModel)EmptyFilter.Handler(model);
            var response = await this.workingHourService.ActivateOrDeactivateTest(model);

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.WorkingHour,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.Now,
                LogDescription = model.Active == true ? $@"<b>{model.ModifiedByName}</b> has Activated status of WorkingHour <br/><strong> '{model.StartDate}'</strong> on {DateTime.UtcNow.AddMinutes(330)}" : $@"<b>{model.ModifiedByName}</b> has Deactivated status of WorkingHour<br/>name: <strong> '{model.StartDate}'</strong> on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            if (response == 0)
            {
                return this.ServerError();
            }

            return this.Success(response);

        }
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] WorkingHourModel model, [FromHeader] LocationHeader header)
        {
            model = (WorkingHourModel)EmptyFilter.Handler(model);
            var response = await this.workingHourService.UpdateAsync(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.WorkingHour,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.Now,
                LogDescription = $"{model.ModifiedByName} has updated <b>WorkingHour</b> of <strong>{model.StartDate}</strong> successfully.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);
        }

    }
}
